// utils.js

function synthetize(o, p /*, init*/) {
    var priv = '_'+p;
    o[p] = function() {
        if (arguments.length === 0) {
            return this[priv];
        }
        else {
            this[priv] = arguments[0];
        }
    };

    if (arguments.length === 3) {o[p](arguments[2]);}
}

function $(q) {return document.querySelector(q);}
function $$(q) {return document.querySelectorAll(q);}

// view.js

/**
 * A view manager. Contains all methods responsible for displaying thigns.
 * @requires Hero
 */
var theView = (function(){
    var buttons,
        uifv, leftMosaic, riverMosaic, raftMosaic, rightMosaic,
        uifOverlay;
    
    var conf = {
        ROWING_TIME : 500,
        LEFT_RAFT_POS : 8,
        RIGHT_RAFT_POS : 42,
        RAFT_POS_UNIT : '%',
        SHIFT_RAFT_MARGIN : '-50px 0 0'
    };
    
    return {

        /**
         * Initialize events, call {@link game#controller} initialization
         */        
        init : function() {
            buttons = $$('button');

            buttons = Array.prototype.map.call(buttons, function(b, inx){
                b.onclick = clickHero;
                b.ID = inx;
                return b;
            });

            // shift raft as a feedback to the user
            $('#river').onmousedown = function() {
                $('#raft').style.margin = conf.SHIFT_RAFT_MARGIN;
            };
            $('#river').onmouseup = function() {$('#raft').style.margin = '';};

            $('#river').onclick = rowTheRaft;

            // clicking the overlay hides it
            $('#conflict').onclick = $('#instructions').onclick 
                = function(evt) {
                    this.style.display = 'none';
                    evt.stopPropagation();
                    uifOverlay.unplug();
                };
            $('#game-over').onclick = function(evt) {
                game.controller.reset();
                this.style.display = 'none';
                evt.stopPropagation();
                uifOverlay.unplug();
            };
            
            // instructions
            $('#info').onclick = function(evt) {
                $('#instructions').style.display = 'block';
                evt.preventDefault();
                uifOverlay = new UIF.View(2, 1, [$('#instructions'), $('#more')]);
                uifOverlay.selectElement(1);
                uifOverlay.plug();
            };

            var rules = $$('.rule');
            var currentRule = 0;
            rules[currentRule].style.display = 'block';

            $("#more").onclick = function(evt) {
                var nextRule = (currentRule + 1) % rules.length;
                rules[currentRule].style.display = '';
                currentRule = nextRule;
                rules[currentRule].style.display = 'block';
                evt.preventDefault();
                evt.stopPropagation();
            };

            // restore
            $('#restore').onclick = function(evt) {
                game.controller.reset();
                evt.preventDefault();
            };

            $('#exit').onclick = function (evt) {
                API.close();
            };
            // init focusing
            initFocusing();

            var kinds = buttons.map(function(b){
                return b.className.toUpperCase()
            });
            game.controller.init(this, kinds);
        },

        /**
         * Place hero's avatar according to it's location
         * @param {Array} array of {@link Hero} instances
         */
        updateHeroes : function(heroes) {

            heroes.forEach(function(hero){
                var btn = buttons[hero.ID];
                btn.parentNode.removeChild(btn);
                var where;
                switch(hero.location()) {
                    case game.places.RAFT:
                        where = $('#raft');
                        break;
                    case game.places.LEFT_SHORE:
                        where = $('#left-group');
                        break;
                    case game.places.RIGHT_SHORE:
                        where = $('#right-group');
                        break;
                    default:
                        throw new Error('Uknown location')
                }
                where.appendChild(btn);
            });

            updateFocusing();
        },
        
        /**
         * Place raft (no animation)
         */
        updateRaft : function() {
            $('#raft').style.left = (
                game.raft.location() === game.places.LEFT_SHORE
                ? conf.LEFT_RAFT_POS : conf.RIGHT_RAFT_POS
            ) + conf.RAFT_POS_UNIT;
        },

        /**
         * @param {Number}
         */
        setCounter : function(value) {
            value = value || '';    // don't display 0
            $('#move-counter').style.opacity = 1;
            $('#move-counter').innerHTML = value;
            setTimeout(function() {
                $('#move-counter').style.opacity = '';
            }, 500)
        },

        /**
         * Display game over screen
         */
        end : function() {
            $('#game-over').style.display = 'block';
            uifOverlay = new UIF.View(1, 1, [$('#game-over')]);
            uifOverlay.selectElement(0);
            uifOverlay.plug();
        },
        
        toString : function() {return '[River IQ Game View Manager]';}
    };
    
    function clickHero(evt) {
        game.controller.moveHeroes([game.heroes[this.ID]]);
        evt.currentTarget.blur();
        // select the raft
        uifv.selectElement([1,1],1)

        evt.stopPropagation();
    }
    
    function rowTheRaft() {
        var problem = game.controller.row();
        if (!problem) {
            raftAnimate();
            return
        }
        //TODO signal problem
        if (problem === game.NO_STEERER || problem === game.RAFT_MOVING) {
            // do nothing
        }
        else {
            $('#conflict .display-heroes').innerHTML 
                = buttons[problem[0].ID].innerHTML
                + buttons[problem[1].ID].innerHTML;
            $('#conflict').style.display = 'block';
            uifOverlay = new UIF.View(1, 1, [$('#conflict')]);
            uifOverlay.selectElement(0);
            uifOverlay.plug();
        }
    }

    function raftAnimate() {
        var from, to, unit = conf.RAFT_POS_UNIT;
        if (game.raft.location() === game.places.LEFT_SHORE) {
            from = conf.LEFT_RAFT_POS;
            to = conf.RIGHT_RAFT_POS;
        }
        else {
            from = conf.RIGHT_RAFT_POS;
            to = conf.LEFT_RAFT_POS;
        }

        var start = new Date().getTime();
        var ti = window.setInterval(function(){
            var interval = new Date().getTime() - start;
            if (interval >= conf.ROWING_TIME) {
                window.clearInterval(ti);
                $('#raft').style.left = to + unit;
                window.setTimeout(function() {
                    game.controller.onReachedTheShore();
                }, 250);
            }
            else {
                $('#raft').style.left = Math.floor(from
                    + (to - from) * interval/conf.ROWING_TIME) + unit;
            }
        }, 50);
    }

    function initFocusing() {
        leftMosaic = new UIF.Mosaic(4, 2),
        raftMosaic = new UIF.Mosaic(1, 2, $('#raft').getElementsByTagName('button'));
        riverMosaic = new UIF.Mosaic(2, 1, [raftMosaic, $('#raft')]),
        rightMosaic = new UIF.Mosaic(4, 2),

        uifv = new UIF.View(2, 3, [$('#exit'), $('#restore'), $('#info'),
            leftMosaic, riverMosaic, rightMosaic]);

        updateFocusing();

        uifv.selectElement(3);  // leftMosaic
        UIF.plug(uifv);
    }

    function updateFocusing() {
        var elLeftGroup = $('#left-group'),
            elRaft = $('#raft'),
            elRightGroup = $('#right-group');
        buttons.forEach(function (btn, inx) {
            switch (btn.parentNode) {
            case elLeftGroup:
                leftMosaic.setKid(inx, btn);
                rightMosaic.setKid(inx, null);
                break;
            case elRaft:
                leftMosaic.setKid(inx, null);
                rightMosaic.setKid(inx, null);
                break;
            case elRightGroup:
                leftMosaic.setKid(inx, null);
                rightMosaic.setKid(inx, btn);
                break;
            }
        });
    }
})();



// Hero.js

/**
 * Build a hero
 * @class Represents a hero
 * @constructor
 * @param Hero kind
 * @param location
 * @param {Integer} ID
 */
function Hero(kind, location, ID) {
    this.ID = ID;
    this._kind = kind;
    this._location = location;
    this._canSteer = this._kind === Hero.MAN || this._kind === Hero.WOMAN
        || this._kind === Hero.COP;
}

Hero.BOY = 'B';
Hero.GIRL = 'G';
Hero.COP = 'C';
Hero.WOMAN = 'W';
Hero.MAN = 'M';
Hero.PRISONER = 'P';

/**
 * Check if there are conflict (see the spec) among heroes in given array
 * @param {Array} heroes ({@link Hero})
 * @return {Array or null} 2-el array of first found conflict or null
 */
Hero.checkForConflict = function(heroes) {
    var people = heroes.map(function(hero){return hero.kind();});

    // at least 2 people for conflict
    if (people.length < 2) return null;

    // prisoner, no cop, others?
    var inxPrisoner = people.indexOf(Hero.PRISONER);
    var inxCop = people.indexOf(Hero.COP);
    if (inxPrisoner !== -1 && inxCop === -1 && people.length > 1) {
        var inxOther = inxPrisoner === 0 ? 1 : 0;
        return [heroes[inxPrisoner], heroes[inxOther]];
    }

    // man, girl, no woman?
    var inxMan = people.indexOf(Hero.MAN);
    var inxGirl = people.indexOf(Hero.GIRL);
    if (
        inxMan !== -1 && inxGirl !== -1 && people.indexOf(Hero.WOMAN) === -1
    ) return [heroes[inxMan], heroes[inxGirl]];

    // woman, boy, no man?
    var inxWoman = people.indexOf(Hero.WOMAN);
    var inxBoy = people.indexOf(Hero.BOY);
    if (
        inxWoman !== -1 && inxBoy !== -1 && people.indexOf(Hero.MAN) === -1
    ) return [heroes[inxWoman], heroes[inxBoy]];

    // no conflicts
    return null;
};

Hero.prototype = {
    constructor : Hero,
    /** 
     * Can steer the raft
     * @type Boolean
     */
    canSteer : function() {return this._canSteer;},
    
    toString : function() {return '[Hero object]';}
};
/** Location getter/setter */
synthetize(Hero.prototype, 'location');
/** Kind getter/setter */
synthetize(Hero.prototype, 'kind');



// game.js

var game = {};

game.places = {
    RAFT : 1,
    LEFT_SHORE : 2,
    RIGHT_SHORE : 3
};

game.heroes = null;

game.NO_STEERER = 7;
game.RAFT_MOVING = 5;

game.controller = (function(){
    var heroes, // alias to game.heroes
        raft,   // alias to game.raft
        view,   // view interface
        _moves = 0;
    return {
        /**
         * Initialize game.heroes and game.raft; connect viewInterface
         * @requires game.raft, Hero
         */
        init : function(viewInterface, kinds) {
            raft = game.raft.init(game.places.LEFT_SHORE);
            heroes = game.heroes = kinds.map(function(kind, inx) {
                return new Hero(Hero[kind], game.places.LEFT_SHORE, inx);
            });
            view = viewInterface;

            return this;
        },
        
        /**
         * Move everything to initial position (left shore)
         */
        reset : function() {
            raft.location(game.places.LEFT_SHORE);
            heroes.forEach(function(hero){
                hero.location(game.places.LEFT_SHORE)});
            view.updateHeroes(game.heroes);
            view.updateRaft();
            _moves = 0;
            view.setCounter(_moves);
        },

        /**
         * Move hero to appropriate location (if possible)
         * @param {Array} array of {@link Hero} instances
         */
        moveHeroes : function (heroes) {
            if (raft.isMoving()) return;
            var update = [];
            for (var i = 0, hero; hero = heroes[i]; ++i) {
                // hero on the raft?
                if (hero.location() === game.places.RAFT) {
                    // onto the same shore as the raft
                    hero.location(raft.location());
                    update.push(hero);
                }
                // hero on the same shore as the raft?
                else if (hero.location() === raft.location()) {
                    if (!raft.isFull()) {
                        hero.location(game.places.RAFT);
                        update.push(hero);
                    }
                }
            }
            view.updateHeroes(update);
        },

        /**
         * Try running the {@link game#raft}
         */
        row : function() {
            if (raft.isMoving()) return game.RAFT_MOVING;
            
            // save at raft?
            var travelConflict = Hero.checkForConflict(raft.passangers);
            if (travelConflict) {
                return travelConflict;
            }

            var these = [];     // these on current shore
            var those = [];     // on the raft and the other shore
            heroes.forEach(function(hero){
                if (hero.location() === raft.location())
                    these.push(hero);
                else
                    those.push(hero);
            });

            // save at this shore?
            var departureConflict = Hero.checkForConflict(these);
            if (departureConflict) {
                return departureConflict;
            }

            // steerer at raft?
            if (!raft.steererAtRaft()) {
                return game.NO_STEERER;
            }

            // save at the other shore and raft?
            var arrivalConflict = Hero.checkForConflict(those);
            if (arrivalConflict) {
                return arrivalConflict;
            }

            // no conflict
            raft.isMoving(true);
            return null;
        },
        
        /**
         * Move heroes from the raft; check if game is over
         */
        onReachedTheShore : function() {
            raft.isMoving(false);
            raft.location(raft.location() === game.places.LEFT_SHORE
                ? game.places.RIGHT_SHORE : game.places.LEFT_SHORE);

            this.moveHeroes(raft.passangers);
            _moves += 1;
            view.setCounter(_moves);
            // endof game - no one on the left shore
            if (
                heroes.filter(function(hero){
                    return hero.location() === game.places.LEFT_SHORE;
                }).length === 0
            ) view.end();
        }
    };

})();

/**
 * @requires Hero
 */
game.raft = {
    /**
     * Heroes who are on the raft
     * @return {Array} 
     */
    get passangers() {
        return game.heroes.filter(function(hero){
            return hero.location() === game.places.RAFT;
        });
    },

    /**
     * Initializes location and isMoving getter/setter (false)
     * @return self
     */
    init : function(place) {
        synthetize(this, 'isMoving', false);
        this.location(place);
        return this;
    },

    /**
     * Is there someone on the raft who can steer it
     * @type Boolean
     */
    steererAtRaft : function() {
        var passangers = this.passangers;
        var len = passangers.length;
        if (len === 0) return false;
        for (var i = 0; i < len; ++i) {
            if (passangers[i].canSteer()) return true;
        }
        return false
    },
    
    /**
     * There are no more places on the raft
     * @type Boolean
     */
    isFull : function() {return this.passangers.length === 2;},

    /** Location getter/setter */
    location : Hero.prototype.location,
    
    toString : function() {return '[Raft object]';}
};
